//=============================================================================
// Sample Application: Lighting (Per Fragment Phong)
//=============================================================================

#include <GL/glew.h>
#include <GL/freeglut.h>
#include "glApplication.h"
#include "glutWindow.h"
#include <iostream>
#include <stdio.h>
#include "glsl.h"
#include <time.h>
#include <math.h>
#include <vector>
#include "TargaImage.h"
#include "Texture.h"
#include "Triangle.h"
#include "MD2Model.h"

//-----------------------------------------------------------------------------

#define texWidth 256
#define texHeight 256

class myWindow : public cwc::glutWindow
{
protected:
	cwc::glShaderManager SM;
	cwc::glShader *shader;
	GLuint ProgramObject;
	clock_t time0,time1;
	float timer010;  // timer counting 0->1->0
	bool bUp;        // flag if counting up or down.
	bool dispSurfaces;
	bool rotate;
	bool software;
	bool animate;
	int frame;
	float rot;

	MD2Model * model;
	std::vector<Triangle*> triangles;
	
	//TEXTURES !
	//TargaImage * positionMap;	//stores surface positions and radii
	//TargaImage * normalMap;		//stores surface normals
	unsigned char * pData;
	unsigned char * nData;
	//float texWidth;
	//float texHeight;

	//TEXTURES 2
	GLuint positionMap;
	GLuint normalMap;
	GLfloat positionBuffer[texWidth][texHeight][4];
	GLfloat normalBuffer[texWidth][texHeight][4];
	GLint positionUniformLoc;
	GLint normalUniformLoc;
	GLint numSurfacesUniformLoc;
	GLint surfaceAttributeLoc;
	//int max_surfaces = 10000;
	Triangle * ground;
public:
	myWindow(){}

	virtual void OnRender(void)
	{
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

		//timer010 = 0.09; //for screenshot!
		glPushMatrix();
		if(rotate){
			rot += .01;
			if(rot > 1) rot = 0;
			//glRotatef(rot*360, 0.5, 1.0f, 0.1f); //timer010
		}
		glRotatef(rot*360, 0.5, 1.0f, 0.1f);
		//glutSolidTeapot(1.0);
		
		// orientate ogro
		glRotatef(-90, 1, 0, 0);
		glRotatef(90, 0, 0, 1);
		if(dispSurfaces){	// diplay surfaces
			model->getSurfaces(&triangles, frame);
			ambientOcclusion();

			for(int i = 0; i < (int)triangles.size(); i++){
				glPushMatrix();
				Triangle * curTriangle = (Triangle *)triangles.at(i);
				glTranslatef(curTriangle->pos[0], curTriangle->pos[1], curTriangle->pos[2]);
				//glRotatef(radToDeg(atan2(curTriangle->normal[1], curTriangle->normal[2])), 1, 0, 0);
				//glRotatef(radToDeg(atan2(curTriangle->normal[1], curTriangle->normal[0])), 0, 0, 1);
				//printf("bnorm: %f", curTriangle
				//setMaterial(curTriangle->bentnormal);
				setMaterial(curTriangle->occlusion);
				//(curTriangle->bentnormal[0], curTriangle->bentnormal[1], curTriangle->bentnormal[2]);
				DrawDisk(curTriangle->radius);
				glPopMatrix();
			}
		} else {	//display model
			if(software){
				model->getSurfaces(&triangles, frame);
				triangles.push_back(ground);
				ambientOcclusion();
				model->render_ambient(&triangles, frame);
				glPushMatrix();
				glTranslatef(ground->pos[0], ground->pos[1], ground->pos[2]);
				DrawDisk(ground->radius);
				glPopMatrix();
				triangles.pop_back();
			} else {
				//int numSurfaces = model->getSurfaces(&positionMap, &normalMap, positionBuffer, normalBuffer, 2);
				//glBindTexture(GL_TEXTURE_2D, positionMap);
				//glutSolidTeapot(30.0);
				if(shader){
					int numSurfaces = model->getSurfaces(&positionMap, &normalMap, positionBuffer, normalBuffer, frame);
					
					shader->begin();
					//set uniform variables
					glUniform1i(positionUniformLoc, positionMap);
					glUniform1i(normalUniformLoc, normalMap);
					glUniform1i(numSurfacesUniformLoc, numSurfaces);
					//glGetUniformiv(ProgramObject, numSurfacesUniformLoc, &numSurfaces);
					//printf("%d surfaces\n", numSurfaces);
				}
				model->render(frame);
				if(shader) shader->end();
			}
		}
		if(animate) {
			frame++;
			if(frame > 100) frame = 2;
		}

		glutSwapBuffers();

		glPopMatrix();

		UpdateTimer();
		Repaint();
	}

	void setMaterial(float * bn){
		GLfloat material_All[] = {bn[0], bn[1], bn[2], 1.0f};

		glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, material_All);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, material_All);
		glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, material_All);
		glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, material_All);
		//glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, material_Se);
	}

	void setMaterial(float val){
		//printf("occlusion: %f", val);
		GLfloat material_All[] = {val, val, val, 1.0f};

		glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, material_All);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, material_All);
		glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, material_All);
		glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, material_All);
		//glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, material_Se);
	}

	void ambientOcclusion(){
		float value, total;
		float d, d2, cosOe, cosOr;
		Triangle * emitter, * receiver;
		float v[3];
		
		for(int r = 1; r < (int)triangles.size(); r++){
			total = 0;
			receiver = triangles.at(r);
			setval(receiver->bentnormal, receiver->normal);
			for(int e = 1; e < (int)triangles.size(); e++){
				if(e == r) continue;
				emitter = triangles.at(e);
				d = distance(emitter->pos, receiver->pos);
				d2 = d*d;
				for(int i = 0; i < 3; i++){
					v[i] = emitter->pos[i] - receiver->pos[i];
				}
				normalize(v);
				cosOr = dot(receiver->normal, v);
				cosOe = dot(emitter->normal, v);
				value = occlusion(d2, cosOr, cosOe, emitter->area);
				
				for(int i = 0; i < 3; i++){
					receiver->bentnormal[i] -= value * v[i]; // update bent normal
				}
				total += value;
			}
			receiver->occlusion = total;
			normalize(receiver->bentnormal);
		}
	}

	float occlusion(float d2, float cosOr, float cosOe, float emitterArea){
		// we assume that emitterArea has already been divided by PI
		return (float)(1.0 - 1.0/sqrt(emitterArea/d2 + 1)) * saturate(cosOe) * saturate(4 * cosOr);
	}

	void DrawDisk(float r){
		GLUquadricObj *disk;
		disk = gluNewQuadric();
		gluQuadricDrawStyle(disk, GLU_FILL);
		gluQuadricNormals(disk, GLU_SMOOTH);
		gluDisk(disk, 0, r, 20, 20);
		gluDeleteQuadric(disk);
	}

	virtual void OnIdle() {}

	// When OnInit is called, a render context (in this case GLUT-Window) 
	// is already available!
	virtual void OnInit() {
		glClearColor(0.5f, 0.5f, 1.0f, 0.0f);
		glShadeModel(GL_SMOOTH);
		glEnable(GL_DEPTH_TEST);
		glEnable(GL_TEXTURE_2D);
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
		shader = SM.loadfromFile("vertexshader.txt","fragmentshader.txt"); // load (and compile, link) from file
		if (shader==0) {
			std::cout << "Error Loading, compiling or linking shader\n";
		} else {
			ProgramObject = shader->GetProgramObject();
			positionUniformLoc = glGetUniformLocation(ProgramObject, "positionMap");
			normalUniformLoc = glGetUniformLocation(ProgramObject, "normalMap");
			numSurfacesUniformLoc = glGetUniformLocation(ProgramObject, "numSurfaces");
			surfaceAttributeLoc = glGetAttribLocation(ProgramObject, "my_Surface");
			//glBindAttribLocation(ProgramObject,  15, "my_Surface");
		}
		shader->disable();

		time0 = clock();
		timer010 = 0.0f;
		rot = 0.0f;
		frame = 2;
		bUp = true;
		dispSurfaces = false;
		rotate = true;
		software = false;
		animate = false;

		ground = new Triangle();
		ground->radius = 30;
		ground->pos[0] = 0;
		ground->pos[1] = 0;
		ground->pos[2] = -30;
		ground->normal[0] = 0;
		ground->normal[1] = 0;
		ground->normal[2] = -1;
		ground->area = ground->radius*ground->radius;

		model = new MD2Model("Ogro.md2", "Ogro.bmp"); //Ogro.bmp
		model->surfaceAttributeLoc = surfaceAttributeLoc;

		//texWidth  = 256;
		//texHeight = 256;
		//nData = new unsigned char[texWidth * texHeight * 4];
		//pData = new unsigned char[texWidth * texHeight * 4];

		//normalMap = new TargaImage(texWidth, texHeight, nData);
		//positionMap = new TargaImage(texWidth, texHeight, pData);

		DemoLight();
	}

	virtual void OnResize(int w, int h) {
		if(h == 0) h = 1;
		float ratio = 1.0f * (float)w / (float)h;

		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		glViewport(0, 0, w, h);
		gluPerspective(45,ratio,1,1000);

		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();
		gluLookAt(0.0, 0.0, -100.0, 
				  0.0, 0.0, 0.0,
				  0.0, 1.0, 0.0);
	}

	virtual void OnClose(void){
		//delete textures
	}
	virtual void OnMouseDown(int button, int x, int y) {}    
	virtual void OnMouseUp(int button, int x, int y) {}
	virtual void OnMouseWheel(int nWheelNumber, int nDirection, int x, int y){}

	virtual void OnKeyDown(int nKey, char cAscii)
	{       
		if (cAscii == 27) // 0x1b = ESC
		{
			this->Close(); // Close Window!
		} 
	};

	virtual void OnKeyUp(int nKey, char cAscii){
		if (cAscii == 's')      // s: Shader
			shader->enable();
		else if (cAscii == 'f') // f: Fixed Function
			shader->disable();
		else if (cAscii == 'r') // r: Toggle Rotate
			rotate = !rotate;
		else if (cAscii == 't') // t: Surfaces/polys
			dispSurfaces = !dispSurfaces;
		else if (cAscii == 'o'){ // s: Run software ambient occlusion
			software = !software;
		}
		else if (cAscii == 'a'){ // s: Run software ambient occlusion
			animate = !animate;
		}
		/*
		else if (cAscii == 'p'){
			model->getSurfaces(positionMap, normalMap, 2);
			for(int i = 0; i < 300; i++){
				unsigned char * pixel = positionMap->data + i * 4;
				printf("pixel[%d] (%d, %c, %u, %f)\n", i, (int)pixel[0], (char)pixel[1], (unsigned int)pixel[2], (float)pixel[3]);
			}
			model->getSurfaces(&triangles, 2);
			for(int i = 0; i < (int)triangles.size(); i++){
				Triangle * curTriangle = (Triangle *)triangles.at(i);
				//printf("tri[%d] (%f, %f, %f, %f)\n", i, curTriangle->pos[0], curTriangle->pos[0], curTriangle->pos[0], curTriangle->radius);
				printf("tri[%d] (%f, %f, %f, %f)\n", i, curTriangle->normal[0], curTriangle->normal[1], curTriangle->normal[2], 0);
			}
		}
		*/
	}

	void UpdateTimer() {
		/*
		if(rotate){
			time1 = clock();
			float delta = static_cast<float>(static_cast<double>(time1-time0)/static_cast<double>(CLOCKS_PER_SEC));
			delta = delta / 4;
			if (delta > 0.00005f){
				time0 = clock();
				if (bUp) {
					timer010 += delta;
					if (timer010>=1.0f) { timer010 = 1.0f; bUp = false;}
				} else {
					timer010 -= delta;
					if (timer010<=0.0f) { timer010 = 0.0f; bUp = true;}
				}
			}
		}
		*/
	}

   void DemoLight(void)
   {
     glEnable(GL_LIGHTING);
     glEnable(GL_LIGHT0);
     glEnable(GL_NORMALIZE);
     
     // Light model parameters:
     // -------------------------------------------
     
     GLfloat lmKa[] = {0.0, 0.0, 0.0, 0.0 };
     glLightModelfv(GL_LIGHT_MODEL_AMBIENT, lmKa);
     
     glLightModelf(GL_LIGHT_MODEL_LOCAL_VIEWER, 1.0);
     glLightModelf(GL_LIGHT_MODEL_TWO_SIDE, 0.0);
     
     // -------------------------------------------
     // Spotlight Attenuation
     
     GLfloat spot_direction[] = {1.0, -1.0, -1.0 };
     GLint spot_exponent = 30;
     GLint spot_cutoff = 180;
     
     glLightfv(GL_LIGHT0, GL_SPOT_DIRECTION, spot_direction);
     glLighti(GL_LIGHT0, GL_SPOT_EXPONENT, spot_exponent);
     glLighti(GL_LIGHT0, GL_SPOT_CUTOFF, spot_cutoff);
    
     GLfloat Kc = 1.0;
     GLfloat Kl = 0.0;
     GLfloat Kq = 0.0;
     
     glLightf(GL_LIGHT0, GL_CONSTANT_ATTENUATION,Kc);
     glLightf(GL_LIGHT0, GL_LINEAR_ATTENUATION, Kl);
     glLightf(GL_LIGHT0, GL_QUADRATIC_ATTENUATION, Kq);
     
     
     // ------------------------------------------- 
     // Lighting parameters:

     GLfloat light_pos[] = {0.0f, 25.0f, 25.0f, 0.0f};
     GLfloat light_Ka[]  = {0.5f, 0.5f, 0.5f, 1.0f};
     GLfloat light_Kd[]  = {0.6f, 0.6f, 0.6f, 1.0f};
     GLfloat light_Ks[]  = {1.0f, 1.0f, 1.0f, 1.0f};

     //glLightfv(GL_LIGHT0, GL_POSITION, light_pos);
     glLightfv(GL_LIGHT0, GL_AMBIENT, light_Ka);
     //glLightfv(GL_LIGHT0, GL_DIFFUSE, light_Kd);
     //glLightfv(GL_LIGHT0, GL_SPECULAR, light_Ks);

     // -------------------------------------------
     // Material parameters:

     GLfloat material_Ka[] = {0.2f, 0.2f, 0.2f, 1.0f};
     GLfloat material_Kd[] = {0.4f, 0.4f, 0.4f, 1.0f};
     GLfloat material_Ks[] = {0.8f, 0.8f, 0.8f, 1.0f};
     GLfloat material_Ke[] = {0.0f, 0.0f, 0.0f, 0.0f};
     GLfloat material_Se = 20.0f;

     glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, material_Ka);
     glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, material_Kd);
     glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, material_Ks);
     glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, material_Ke);
     glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, material_Se);
   }
};

//-----------------------------------------------------------------------------

class myApplication : public cwc::glApplication
{
public:
	virtual void OnInit() {std::cout << "Hello World!\n"; }
};

//-----------------------------------------------------------------------------

int main(void)
{
	myApplication*  pApp = new myApplication;
	myWindow* myWin = new myWindow();

	pApp->run();
	delete pApp;
	return 0;
}

//-----------------------------------------------------------------------------

